/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, node: true, plusplus: true, devel: true, nomen: true, indent: 4, expr:true */
/*global require */
/*global beforeEach: false, afterEach: false, it: false, describe: false, expect: false*/

"use strict";

var _ = require("underscore"),
    docinfo = require("./resources/quickDocInfo.json"),
    fs = require("fs"),
    path = require("path"),
    sinon = require("sinon"),
    BoundsUtils = require("shared/BoundsUtils"),
    DocinfoUtils = require("shared/DocinfoUtils"),
    Errors = require("../lib/errors"),
    FilePathSanitizer = require("shared/FilePathSanitizer"),
    Headlights = require("../lib/headlights"),
    MaxPixels = require("shared/MaxPixels"),
    Constants = require("shared/Constants"),
    PSDialogs = require("../lib/ps-dialogs"),
    PSEventStrings = require("../lib/ps-event-strings"),
    Q = require("q"),
    QuickExport = require("../lib/quick-export");

describe("QuickExport", function () {

    var sandbox = sinon.sandbox.create(),
        quickExport = null,
        destFolder = "/Users/Batman/Desktop",
        docinfoWithSingleSelection = _(docinfo).chain().clone().assign({
            _selectionById: [ 7 ]
        }).value(),
        docinfoWithMultipleSelection = _(docinfo).chain().clone().assign({
            _selectionById: [ 7, 13 ]
        }).value(),
        docinfoWithMultipleNestedSelection = _(docinfo).chain().clone().assign({
            // 27 contains only 28 and 29, and both are selected, so they won't be exported.
            _selectionById: [ 7, 20, 27, 28, 29, 13 ]
        }).value(),
        docinfoWithMultiplePartiallyEmptySelection = _(docinfo).chain().clone().assign({
            _selectionById: [ 13, 14 ] // 13 is a non-empty layer, 14 is an empty layer
        }).value(),
        docinfoWithMultipleEmptySelection = _(docinfo).chain().clone().assign({
            _selectionById: [ 14, 15 ]
        }).value(),
        docinfoWithMultipleOutOfDocBoundsSelection = _(docinfo).chain().clone().assign({
            _selectionById: [ 16, 17 ]
        }).value(),
        docinfoWithMultipleEmptyAndOutOfDocBoundsSelection = _(docinfo).chain().clone().assign({
            _selectionById: [ 14, 15, 16, 17 ]
        }).value(),
        docinfoWithEmptyArtboardNoChildren = _(docinfo).chain().clone().assign({
            _selectionById: [ 31 ]
        }).value(),
        docinfoWithEmptyArtboardEmptyChildren = _(docinfo).chain().clone().assign({
            _selectionById: [ 32 ]
        }).value(),
        docinfoWithClippedByDocSelection = _(docinfo).chain().clone().assign({
            _selectionById: [ 18 ]
        }).value(),
        docinfoWithSameNameLayers = function () {
            var resultDocinfo = _(docinfo).chain().clone().assign({
                layers: [
                    _(docinfo.layers[0]).chain().clone().assign({
                        id: 1,
                        name: "samename"
                    }).value(),
                    _(docinfo.layers[1]).chain().clone().assign({
                        id: 2,
                        name: "samename"
                    }).value(),
                    _(docinfo.layers[1]).chain().clone().assign({
                        id: 3,
                        name: "samename-1"
                    }).value()
                ],
                _selectionById: [ 1, 2, 3 ]
            }).value();
            return resultDocinfo;
        }();

    var psEvent = {};
    psEvent[PSEventStrings.QUICK_EXPORT_DOCUMENT] = {};

    //since join is basically string operators i am just going to stub this with a simple array join.
    //we are not doing anything strange in these tests that needs edge cases covered
    //after we ugprade to node 12.xx we can simply stub it with path.posix.join
    //also update in fileDialog.text.js
    var tempJoin = function() {
        var args = Array.prototype.slice.call(arguments);
        //for our unit tests we want to just join with a posix path and avoid windows
        return args.join("/");
    };

    beforeEach(function () {
        var quickExportOptions = {
            headlights: new Headlights(),
            psDialogs: {
                alert: sinon.stub(),
                promptForFile: sinon.stub(),
                promptForFolder: sinon.stub()
            },
            requestAccurateBoundsFunction: sinon.stub().returns(Q.resolve()),
            exportComponentsFunction: sinon.stub().returns(Q.resolve([])),
            metadataProvider: {
                readMetadata: sinon.stub(),
                writeMetadata: sinon.stub()
            }
        };
        quickExport = new QuickExport(quickExportOptions);
        quickExport._headlights.logEvent = sinon.stub();
        quickExport._headlights.accumulateData = sinon.stub();
        quickExport._headlights.logAccumulatedData = sinon.stub();
        quickExport._headlights.logData = sinon.stub();
        sandbox.stub(path, "join", tempJoin);
    });

    afterEach(function () {
        quickExport = null;
        sandbox.restore();
    });

    it("should perform quick export workflow when run", function () {
        quickExport._performWorkflow = sinon.stub().returns(Q.resolve());
        return expect(quickExport.run(psEvent)).to.eventually.be.fulfilled
            .finally(function () {
                expect(quickExport._performWorkflow).to.have.been.calledOnce;
            });
    });

    it("should show user facing errors in an alert", function () {
        quickExport._performWorkflow = sinon.stub().returns(Q.reject(new Errors.EmptySelectionError()));
        return expect(quickExport.run(psEvent)).to.eventually.be.fulfilled
            .finally(function () {
                expect(quickExport._psDialogs.alert).to.have.been.calledOnce;
            });
    });

    it("should reject the returned promise if the quick export workflow fails unexpectedly", function () {
        quickExport._performWorkflow = sinon.stub().returns(Q.reject("Something went wrong!"));
        return expect(quickExport.run(psEvent)).to.eventually.be.rejectedWith("Something went wrong!");
    });

    it("_logAssetSummary should log appropriate headlights data", function () {
        quickExport._logAssetSummary("type", "count", "sec");

        expect(quickExport._headlights.accumulateData.callCount).to.equal(4);
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.EXPORTTYPE, quickExport._headlights.QUICK);
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.ITEMTYPE, "type");
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.ITEMCOUNT, "count");
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.SECONDS, "sec");
        expect(quickExport._headlights.logAccumulatedData).to.have.been.calledWith(quickExport._headlights.EXPORT_SUMMARY_GROUP);
    });

    it("_logDataForOneAsset should log appropriate headlights data for jpg", function () {
        var event = { fileType: "jpg", quality: 12 },
            bounds = {width: function() {return 33;}, height: function() {return 44;}};

        quickExport._logDataForOneAsset(event, bounds, "itemType", {});

        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.EXPORTTYPE, quickExport._headlights.QUICK);
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.ITEMTYPE, "itemType");
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.REQUESTED_WIDTH, 33);
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.REQUESTED_HEIGHT, 44);
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.FORMAT, "jpg");
        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.QUALITY, 12);
        expect(quickExport._headlights.logAccumulatedData).to.have.been.calledWith(quickExport._headlights.EXPORT_ASSETS_GROUP);
    });

    it("_logDataForOneAsset should log appropriate headlights data for png8", function () {
        var event = { fileType: "png", quality: 8 },
            bounds = {width: function() {return 33;}, height: function() {return 44;}};

        quickExport._logDataForOneAsset(event, bounds, "itemType", {});

        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.FORMAT, "png8");
        expect(quickExport._headlights.accumulateData).to.not.have.been.calledWith(quickExport._headlights.QUALITY);
    });

    it("_logDataForOneAsset should log appropriate headlights data for png-248", function () {
        var event = { fileType: "png-24", quality: 99 },
            bounds = {width: function() {return 33;}, height: function() {return 44;}};

        quickExport._logDataForOneAsset(event, bounds, "itemType", {});

        expect(quickExport._headlights.accumulateData).to.have.been.calledWith(quickExport._headlights.FORMAT, "png24");
        expect(quickExport._headlights.accumulateData).to.not.have.been.calledWith(quickExport._headlights.QUALITY);
    });

    it("should log appropriate headlights events for exporting document", function () {
        var event = {name: PSEventStrings.QUICK_EXPORT_DOCUMENT},
            bounds = {fake: "boundsObj"},
            docComponent = {extension: "jpg",
                            quality: "32",
                            document: {bounds: bounds}
                        };

        sandbox.stub(quickExport, "_logAssetSummary");
        sandbox.stub(quickExport, "_logDataForOneAsset");

        quickExport._logExportHeadlights(event, [docComponent], true, 99.9);

        expect(quickExport._logAssetSummary).to.have.been.calledWith(quickExport._headlights.DOCUMENT, 1, 99.9);
        expect(quickExport._logDataForOneAsset).to.have.been.calledWith(event, bounds, quickExport._headlights.DOCUMENT);
        expect(quickExport._headlights.logEvent).to.have.been.calledWith(quickExport._headlights.CREMA_FUNNEL, quickExport._headlights.QUICK_EXPORT_FOR_DOCUMENT_DONE);
    });

    it("should log appropriate headlights events for exporting document with artboards", function () {
        var event = {name: PSEventStrings.QUICK_EXPORT_DOCUMENT},
            components = [
                {layer: {bounds: "bounds1"}},
                {layer: {bounds: "bounds2"}}
            ];

        sandbox.stub(quickExport, "_logAssetSummary");
        sandbox.stub(quickExport, "_logDataForOneAsset");

        quickExport._logExportHeadlights(event, components, false, 12.3);

        expect(quickExport._logAssetSummary).to.have.been.calledWith(quickExport._headlights.ARTBOARD, 2, 12.3);
        expect(quickExport._logDataForOneAsset).to.have.been.calledWith(event, "bounds1", quickExport._headlights.ARTBOARD);
        expect(quickExport._logDataForOneAsset).to.have.been.calledWith(event, "bounds2", quickExport._headlights.ARTBOARD);
        expect(quickExport._headlights.logEvent).to.have.been.calledWith(quickExport._headlights.CREMA_FUNNEL, quickExport._headlights.QUICK_EXPORT_FOR_DOCUMENT_ARTBOARDS_DONE + 2);
    });

    it("should log appropriate headlights events for exporting selection", function () {
        var event = {name: PSEventStrings.QUICK_EXPORT_SELECTION},
            components = [
                {layer: {bounds: "bounds1"}},
                {layer: {bounds: "bounds2"}},
                {layer: {bounds: "bounds3"}}
            ];

        sandbox.stub(quickExport, "_logAssetSummary");
        sandbox.stub(quickExport, "_logDataForOneAsset");

        quickExport._logExportHeadlights(event, components, false, 32.1);

        expect(quickExport._logAssetSummary).to.have.been.calledWith(quickExport._headlights.SELECTION, 3, 32.1);
        expect(quickExport._logDataForOneAsset).to.have.been.calledWith(event, "bounds1", quickExport._headlights.SELECTION);
        expect(quickExport._logDataForOneAsset).to.have.been.calledWith(event, "bounds2", quickExport._headlights.SELECTION);
        expect(quickExport._logDataForOneAsset).to.have.been.calledWith(event, "bounds3", quickExport._headlights.SELECTION);
        expect(quickExport._headlights.logEvent).to.have.been.calledWith(quickExport._headlights.CREMA_FUNNEL, quickExport._headlights.QUICK_EXPORT_FOR_SELECTION_DONE + 3);
    });

    it("should export the document in the workflow", function () {
        sandbox.stub(quickExport, "_exportDocument").returns(Q.resolve());
        sandbox.stub(DocinfoUtils, "getArtboards"). returns([]);

        var event = {
            name: PSEventStrings.QUICK_EXPORT_DOCUMENT
        };

        return expect(quickExport._performWorkflow(event)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._exportDocument).to.have.been.calledOnce;
                expect(quickExport._headlights.logEvent).to.have.been.calledWith(
                    quickExport._headlights.CREMA_FUNNEL,
                    quickExport._headlights.QUICK_EXPORT_FOR_DOCUMENT_START
                );
            });
    });

    it("should export the document's artboards in the workflow", function () {
        var event = {
                name: PSEventStrings.QUICK_EXPORT_DOCUMENT
            },
            docinfo = "fakeinfo",
            artboards = ["fakeArtboard"];

        sandbox.stub(quickExport, "_exportDocument").returns(Q.resolve());
        sandbox.stub(quickExport, "_exportLayers").returns(Q.resolve());
        sandbox.stub(DocinfoUtils, "getArtboards"). returns(artboards);

        return expect(quickExport._performWorkflow(event, docinfo)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._exportLayers).to.be.calledOnce;
                expect(quickExport._exportLayers).to.be.calledWith(event, docinfo, artboards);
                expect(quickExport._headlights.logEvent).to.have.been.calledWith(
                    quickExport._headlights.CREMA_FUNNEL,
                    quickExport._headlights.QUICK_EXPORT_FOR_DOCUMENT_ARTBOARDS_START
                );
            });
    });

    it("should export the selection in the workflow", function () {
        sandbox.stub(quickExport, "_exportLayers").returns(Q.resolve());
        sandbox.stub(DocinfoUtils, "getLayersForExport").returns(["a", "b", "c"]);

        var event = {
                name: PSEventStrings.QUICK_EXPORT_SELECTION
            },
            docinfo = {_selectionById: ["a", "b", "c"]};


        return expect(quickExport._performWorkflow(event, docinfo)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._exportLayers).to.have.been.calledOnce;
                expect(quickExport._headlights.logEvent).to.have.been.calledOnce;
                expect(quickExport._headlights.logEvent).to.have.been.calledWith(
                    quickExport._headlights.CREMA_FUNNEL,
                    quickExport._headlights.QUICK_EXPORT_FOR_SELECTION_START + "3"
                );
            });
    });

    it("should log when some of selection isn't exportable", function () {
        sandbox.stub(quickExport, "_exportLayers").returns(Q.resolve());
        sandbox.stub(DocinfoUtils, "getLayersForExport").returns(["a"]);

        var event = {
                name: PSEventStrings.QUICK_EXPORT_SELECTION
            },
            docinfo = {_selectionById: ["a", "b", "c"]};

        return expect(quickExport._performWorkflow(event, docinfo)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._headlights.logEvent).to.have.been.calledTwice;
                expect(quickExport._headlights.logEvent).to.have.been.calledWith(
                    quickExport._headlights.CREMA_FUNNEL,
                    quickExport._headlights.QUICK_EXPORT_FOR_SELECTION_START + "3"
                );
                expect(quickExport._headlights.logEvent).to.have.been.calledWith(
                    quickExport._headlights.CREMA_ACTION,
                    quickExport._headlights.EXPORTING_FEWER_THAN_SELECTED_QUICK
                );
            });
    });

    it("docinfoutils getArtboards returns the layers that are artboards", function () {
        var docInfoWithArtboards = {
                    layers: [{name: "plain1"},
                             {name: "artboard1", artboard: {top: 1}},
                             {name: "artboard2", artboard: {top: 2}}
                            ]
                },
            artboards = DocinfoUtils.getArtboards(docInfoWithArtboards);

        expect(artboards.length).to.equal(2);
    });

    it("should export components by calling the passed-in function", function () {
        var components = [
            {
                documentId: 3,
                layerId: 7,
                extension: "png",
                path: "/Users/Batman/Desktop/batman.png"
            },
            {
                documentId: 3,
                layerId: 13,
                extension: "png",
                path: "/Users/Batman/Desktop/robin.png"
            }
        ];

        return expect(quickExport._exportComponents(components)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._exportComponentsFunction).to.have.been.calledWith(components);
            });
    });

    it("should not call readMetadata if no components request meta data added", function () {
        var components = [
            {
                documentId: 3,
                layerId: 7,
                extension: "png",
                path: "/Users/Batman/Desktop/batman.png"
            },
            {
                documentId: 3,
                layerId: 13,
                extension: "png",
                path: "/Users/Batman/Desktop/robin.png"
            }
        ];

        return expect(quickExport._exportComponents(components)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._metadataProvider.readMetadata).to.have.not.been.called;
            });
    });

     it("should call readMetadata just once even if multiple components request data", function () {
        var components = [
            {
                documentId: 3,
                layerId: 7,
                extension: "png",
                path: "/Users/Batman/Desktop/batman.png",
                metadataType: 1
            },
            {
                documentId: 3,
                layerId: 13,
                extension: "png",
                path: "/Users/Batman/Desktop/robin.png",
                metadataType: 1
            }
        ];

        quickExport._metadataProvider.readMetadata.returns(Q.resolve());
        quickExport._metadataProvider.writeMetadata.returns([Q.resolve()]);

        return expect(quickExport._exportComponents(components)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._metadataProvider.readMetadata).to.have.been.calledOnce;
            });
    });


    it("should call write meta data once per component with provided data", function () {
        var components = [
            {
                documentId: 3,
                layerId: 7,
                extension: "png",
                path: "/Users/Batman/Desktop/batman.png",
                metadataType: 1
            },
            {
                documentId: 3,
                layerId: 13,
                extension: "png",
                path: "/Users/Batman/Desktop/robin.png",
                metadataType: 1
            }
        ], data = "hello world";

        quickExport._metadataProvider.readMetadata.returns(Q.resolve(data));
        quickExport._metadataProvider.writeMetadata.returns([Q.resolve()]);

        return expect(quickExport._exportComponents(components)).to.eventually.be.fulfilled
            .then(function () {
                expect(quickExport._metadataProvider.writeMetadata).to.have.been.calledTwice;
                var firstCall = quickExport._metadataProvider.writeMetadata.firstCall.args,
                    secondCall = quickExport._metadataProvider.writeMetadata.secondCall.args;

                expect(firstCall[0]).to.equal(components[0].path);
                expect(secondCall[0]).to.equal(components[1].path);

                expect(firstCall[1]).to.equal(data);
                expect(secondCall[1]).to.equal(data);

            });
    });


    describe("building components", function () {
        var documentId = 123,
            layerId = null,
            event = {
                name: PSEventStrings.QUICK_EXPORT_DOCUMENT,
                destFolder: destFolder,
                fileType: "jpg",
                quality: 42
            };

        it("should include quality if available when building components", function () {
            var component = quickExport._buildComponent(event, documentId, layerId, {}, destFolder);

            expect(component.quality).to.be.equal(42);
        });

        it("should include useICCProfile if available when building components", function () {
            var updatedEvents = _.extend({}, event, {sRGB: true}),
                component = quickExport._buildComponent(updatedEvents, documentId, layerId, {}, destFolder);

            expect(component.useICCProfile).to.be.equal(Constants.SRGB_COLOR_PROFILE);
        });

        it("should include metadata if available when building components", function () {
            var updatedEvents = _.extend({}, event, {metadata: Constants.META_DATA.COPYRIGHTANDCONTACT}),
                component = quickExport._buildComponent(updatedEvents, documentId, layerId, {}, destFolder);

            expect(component.metadataType).to.be.equal(Constants.META_DATA.COPYRIGHTANDCONTACT);
        });

        it("should not include a metadata property if set to none", function () {
            var updatedEvents = _.extend({}, event, {metadata: Constants.META_DATA.NONE}),
                component = quickExport._buildComponent(updatedEvents, documentId, layerId, {}, destFolder);

            expect(component.metadataType).to.be.undefined;
        });

        it("should clamp scale using max pixels when building components", function () {
            sandbox.stub(MaxPixels, "getMaxPixels").returns(1 * 1000 * 1000);

            var component = quickExport._buildComponent(event, documentId, layerId, {
                // Source image is 2000 x 1000
                top: 1000,
                bottom: 2000,
                left: 2000,
                right: 4000
            }, destFolder);

            // sqrt(1000000 / (2000 * 1000)) = 0.707, rounded down to 0.70
            expect(component.scale).to.be.equal(0.70);
            expect(quickExport._headlights.logEvent).to.have.been.calledWith(quickExport._headlights.CREMA_WARNING, quickExport._headlights.MAX_EXCEEDED_SCALE);
            expect(quickExport._headlights.logData).to.have.been.calledWith(quickExport._headlights.MAX_EXCEEDED_GROUP, quickExport._headlights.MAX_EXCEEDED_SCALE, 1);

        });
    });

    describe('without a save prompt', function () {
        beforeEach(function () {
            sandbox.spy(BoundsUtils, "ensureAccurateBoundsForLayers");
            sandbox.spy(quickExport, "_buildComponent");
            sandbox.stub(quickExport, "_exportComponents");
            sandbox.stub(quickExport, "_logAssetSummary");
            sandbox.stub(quickExport, "_logDataForOneAsset");
            sandbox.stub(quickExport, "_logExportHeadlights");
        });

        it("should export the document without a save prompt", function () {
            var event = {
                name: PSEventStrings.QUICK_EXPORT_DOCUMENT,
                destFolder: destFolder,
                fileType: "png"
            };

            return expect(quickExport._exportDocument(event, docinfo)).to.eventually.be.fulfilled
                .then(function () {
                    expect(BoundsUtils.ensureAccurateBoundsForLayers).to.not.have.been.called;
                    expect(quickExport._psDialogs.promptForFile).to.not.have.been.called;
                    expect(quickExport._psDialogs.promptForFolder).to.not.have.been.called;
                    expect(quickExport._buildComponent).to.have.been.calledOnce;
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            extension: "png",
                            path: "/Users/Batman/Desktop/gotham-skyline.png"
                        }
                    ]);
                });
        });

        it("should export a single layer selection without a save prompt", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithSingleSelection);

            return expect(quickExport._exportLayers(event, docinfoWithSingleSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(BoundsUtils.ensureAccurateBoundsForLayers).to.have.been.calledWith(sinon.match(function (layers) {
                        return layers.length == docinfoWithSingleSelection._selectionById.length;
                    }), docinfoWithSingleSelection);
                    expect(quickExport._psDialogs.promptForFile).to.not.have.been.called;
                    expect(quickExport._psDialogs.promptForFolder).to.not.have.been.called;
                    expect(quickExport._buildComponent).to.have.been.calledOnce;
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            layerId: 7,
                            extension: "png",
                            path: "/Users/Batman/Desktop/batman.png"
                        }
                    ]);
                });
        });

        it("should export a multiple layer selection without a save prompt", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithMultipleSelection);

            return expect(quickExport._exportLayers(event, docinfoWithMultipleSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(BoundsUtils.ensureAccurateBoundsForLayers).to.have.been.calledWith(sinon.match(function (layers) {
                        return layers.length == docinfoWithMultipleSelection._selectionById.length;
                    }), docinfoWithMultipleSelection);
                    expect(quickExport._psDialogs.promptForFile).to.not.have.been.called;
                    expect(quickExport._psDialogs.promptForFolder).to.not.have.been.called;
                    expect(quickExport._buildComponent).to.have.been.calledTwice;
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            layerId: 7,
                            extension: "png",
                            path: "/Users/Batman/Desktop/batman.png"
                        },
                        {
                            documentId: 3,
                            layerId: 13,
                            extension: "png",
                            path: "/Users/Batman/Desktop/robin.png"
                        }
                    ]);
                });
        });

        it("should suffix filenames numerically if layers have the same name", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithSameNameLayers);

            return expect(quickExport._exportLayers(event, docinfoWithSameNameLayers, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            layerId: 1,
                            extension: "png",
                            // This layer was originally called "samename". Name was preserved.
                            path: "/Users/Batman/Desktop/samename.png"
                        },
                        {
                            documentId: 3,
                            layerId: 2,
                            extension: "png",
                            // This layer was originally called "samename". Name had to be changed.
                            path: "/Users/Batman/Desktop/samename-2.png"
                        },
                        {
                            documentId: 3,
                            layerId: 3,
                            extension: "png",
                            // This layer was originally called "samename-1". Name was preserved.
                            path: "/Users/Batman/Desktop/samename-1.png"
                        }
                    ]);
                });
        });

        it("should only export layers if they are selected and all their siblings are not also selected", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithMultipleNestedSelection);

            return expect(quickExport._exportLayers(event, docinfoWithMultipleNestedSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(BoundsUtils.ensureAccurateBoundsForLayers).to.have.been.calledWith(sinon.match(function (layers) {
                        return layers.length == 4;
                    }), docinfoWithMultipleNestedSelection);
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            layerId: 7,
                            extension: "png",
                            path: "/Users/Batman/Desktop/batman.png"
                        },
                        {
                            documentId: 3,
                            layerId: 20,
                            extension: "png",
                            path: "/Users/Batman/Desktop/cape.png"
                        },
                        {
                            documentId: 3,
                            layerId: 27,
                            extension: "png",
                            path: "/Users/Batman/Desktop/ears.png"
                        },
                        {
                            documentId: 3,
                            layerId: 13,
                            extension: "png",
                            path: "/Users/Batman/Desktop/robin.png"
                        }
                    ]);
                });
        });

        it("should generate an error if all selected layers are empty", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithMultipleEmptySelection);

            return expect(quickExport._exportLayers(event, docinfoWithMultipleEmptySelection, layers)).to.eventually.be.rejected
                .then(function (e) {
                    expect(e instanceof Errors.EmptySelectionError).to.be.true;
                    expect(quickExport._headlights.logEvent.callCount).to.equal(2);
                    expect(quickExport._headlights.logEvent).to.have.been.calledWith( quickExport._headlights.CREMA_ACTION, quickExport._headlights.LAYER_EMPTY_QUICK_EXP);
                });
        });

        it("should generate an error if all selected layers are out of bounds", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithMultipleOutOfDocBoundsSelection);

            return expect(quickExport._exportLayers(event, docinfoWithMultipleOutOfDocBoundsSelection, layers)).to.eventually.be.rejected
                .then(function (e) {
                    expect(e instanceof Errors.EmptySelectionError).to.be.true;
                    expect(quickExport._headlights.logEvent.callCount).to.equal(2);
                    expect(quickExport._headlights.logEvent).to.have.been.calledWith( quickExport._headlights.CREMA_ACTION, quickExport._headlights.LAYER_OUTSIDE_DOC_QUICK_EXP);
                });
        });

        it("should generate an error if all selected layers are either empty or out of bounds", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithMultipleEmptyAndOutOfDocBoundsSelection);

            return expect(quickExport._exportLayers(event, docinfoWithMultipleEmptyAndOutOfDocBoundsSelection, layers)).to.eventually.be.rejected
                .then(function (e) {
                    expect(e instanceof Errors.EmptySelectionError).to.be.true;
                });
        });

        it("should log headlights event when layer is clipped by doc bounds", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithClippedByDocSelection);

            return expect(quickExport._exportLayers(event, docinfoWithClippedByDocSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(quickExport._headlights.logEvent.callCount).to.equal(1);
                    expect(quickExport._headlights.logEvent).to.have.been.calledWith( quickExport._headlights.CREMA_ACTION, quickExport._headlights.LAYER_CLIPPEDBY_DOC_QUICK_EXP);
                });
        });

        it("should log headlights event when artboard has no child layers", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport (docinfoWithEmptyArtboardNoChildren);

            return expect(quickExport._exportLayers(event, docinfoWithEmptyArtboardNoChildren, layers)).to.eventually.be.rejected
                .then(function () {
                    expect(quickExport._headlights.logEvent.callCount).to.equal(1);
                    expect(quickExport._headlights.logEvent).to.have.been.calledWith( quickExport._headlights.CREMA_ACTION, quickExport._headlights.ARTBOARD_EMPTY_QUICK_EXP);
                });
        });

        it("should log headlights event when artboard child layers are all empty", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport (docinfoWithEmptyArtboardEmptyChildren);

            return expect(quickExport._exportLayers(event, docinfoWithEmptyArtboardEmptyChildren, layers)).to.eventually.be.rejected
                .then(function () {
                    expect(quickExport._headlights.logEvent.callCount).to.equal(1);
                    expect(quickExport._headlights.logEvent).to.have.been.calledWith( quickExport._headlights.CREMA_ACTION, quickExport._headlights.ARTBOARD_EMPTY_QUICK_EXP);
                });
        });

        it("should not generate an error if some selected layers are non-empty", function () {
            var event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png"
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithMultiplePartiallyEmptySelection);
            quickExport._headlights.logEvent = sinon.stub();

            return expect(quickExport._exportLayers(event, docinfoWithMultiplePartiallyEmptySelection, layers)).to.eventually.be.fulfilled
                .then(function (e) {
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            layerId: 13,
                            extension: "png",
                            path: "/Users/Batman/Desktop/robin.png"
                        }
                    ]);
                    expect(quickExport._headlights.logEvent).to.have.been.calledWith( quickExport._headlights.CREMA_ACTION, quickExport._headlights.LAYER_EMPTY_QUICK_EXP);
                });
        });
    });

    describe("handle errors from exportComponentsFunction", function() {
        var components = [
            {
                documentId: 3,
                layerId: 7,
                extension: "png",
                path: "/Users/Batman/Desktop/batman.png"
            },
            {
                documentId: 3,
                layerId: 13,
                extension: "png",
                path: "/Users/Batman/Desktop/robin.png"
            }
        ];

        it("should throw a FSWrite error if the exportComponentsFunction returns a write error", function() {
            quickExport._exportComponentsFunction.returns(Q.resolve([
                {
                    state: "rejected",
                    reason: new Error("SomeKindOfWriteError")
                }
            ]));
            sandbox.stub(Errors, "FSWriteError");

            return expect(quickExport._exportComponents(components)).to.eventually.be.rejected
                .then(function (e) {
                    expect(e instanceof Errors.FSWriteError).to.be.true;
                    expect(quickExport._headlights.logEvent).to.have.been.calledOnce;
                    expect(quickExport._headlights.logEvent).to.have.been.calledWith( quickExport._headlights.CREMA_ACTION, quickExport._headlights.WRITEERROR_ON_QUICK_EXPORT);
                });
        });

        it("should not throw a FSWrite error if the exportComponentsFunction returns some other error", function() {
            quickExport._exportComponentsFunction.returns(Q.reject([
                {
                    state: "rejected",
                    reason: new Error("SomeRandomError")
                }
            ]));
            return expect(quickExport._exportComponents(components)).to.eventually.be.rejected
                .then(function (e) {
                    expect(e instanceof Errors.FSWriteError).to.be.false;
                });
        });
    });

    describe('with a save prompt', function () {
        beforeEach(function () {
            sandbox.spy(quickExport, "_buildComponent");
            sandbox.spy(quickExport, "_exportComponents");
            sandbox.stub(BoundsUtils, "ensureAccurateBoundsForLayers");
            sandbox.stub(FilePathSanitizer, "sanitize").returnsArg(0);
            sandbox.stub(quickExport, "_logAssetSummary");
            sandbox.stub(quickExport, "_logDataForOneAsset");
            sandbox.stub(quickExport, "_logExportHeadlights");

            sandbox.stub(quickExport._UserSettings, "setCachedValue");

        });

        it("should export the document with a file save prompt", function () {
            // Pretend the user will select a different path.
            var expectedFile = path.basename(docinfo.file, path.extname(docinfo.file)),
                expectedFileType = "png",
                expectedPath = destFolder + "/" + expectedFile + "." + expectedFileType,
                event = {
                    name: PSEventStrings.QUICK_EXPORT_DOCUMENT,
                    destFolder: destFolder,
                    fileType: expectedFileType,
                    promptForSaveLocation: true
                };
            sandbox.stub(quickExport._UserSettings, "get").returns(undefined);
            quickExport._psDialogs.promptForFile.returns(Q.resolve(expectedPath));

            return expect(quickExport._exportDocument(event, docinfo)).to.eventually.be.fulfilled
                .then(function () {
                    expect(quickExport._requestAccurateBoundsFunction).to.not.have.been.called;
                    expect(FilePathSanitizer.sanitize).to.have.been.calledWith(expectedFile);
                    expect(quickExport._psDialogs.promptForFile).to.have.been.calledOnce;
                    expect(quickExport._psDialogs.promptForFile).to.have.been.calledWith(destFolder);
                    expect(quickExport._buildComponent).to.have.been.calledOnce;
                    expect(quickExport._UserSettings.get).to.be.called;
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            extension: expectedFileType,
                            path: expectedPath
                        }
                    ]);
                });
        });

        it("should export the document with a file save prompt using the stored user setting", function () {
                // Pretend the user will select a different path.
                var expectedFile = path.basename(docinfo.file, path.extname(docinfo.file)),
                    expectedFileType = "png",
                    expectedPath = destFolder + "/" + expectedFile + "." + expectedFileType,
                    event = {
                        name: PSEventStrings.QUICK_EXPORT_DOCUMENT,
                        destFolder: destFolder,
                        fileType: expectedFileType,
                        promptForSaveLocation: true
                    };

                quickExport._psDialogs.promptForFile.returns(Q.resolve(expectedPath));
                sandbox.stub(quickExport._UserSettings, "get").returns("new/path");
                return expect(quickExport._exportDocument(event, docinfo)).to.eventually.be.fulfilled
                    .then(function () {
                        expect(quickExport._psDialogs.promptForFile).to.have.been.calledWith("new/path");
                    });
            });

        it("should export a single layer selection with a file save prompt", function () {
            // Pretend the user will select a different path.
            var expectedFile = _(docinfoWithSingleSelection.layers).first().name,
                expectedFileType = "png",
                expectedPath = destFolder + "/" + expectedFile + "." + expectedFileType,
                event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: expectedFileType,
                    promptForSaveLocation: true
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithSingleSelection);

            quickExport._psDialogs.promptForFile.returns(Q.resolve(expectedPath));

            return expect(quickExport._exportLayers(event, docinfoWithSingleSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(quickExport._UserSettings.setCachedValue).to.be.calledWith("/Users/Batman/Desktop/gotham-skyline.psd", destFolder);
                    expect(BoundsUtils.ensureAccurateBoundsForLayers).to.have.been.calledWith(sinon.match(function (layers) {
                        return layers.length == docinfoWithSingleSelection._selectionById.length;
                    }), docinfoWithSingleSelection, quickExport._requestAccurateBoundsFunction);
                    expect(FilePathSanitizer.sanitize).to.have.been.calledWith(expectedFile);
                    expect(quickExport._psDialogs.promptForFile).to.have.been.calledOnce;
                    expect(quickExport._buildComponent).to.have.been.calledOnce;
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            layerId: 7,
                            extension: expectedFileType,
                            path: expectedPath
                        }
                    ]);
                });
        });


        it("should use the user setting for the dest folder if exist", function () {
            // Pretend the user will select a different path.
            var expectedFile = _(docinfoWithSingleSelection.layers).first().name,
                expectedFileType = "png",
                expectedPath = destFolder + "/" + expectedFile + "." + expectedFileType,
                event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: expectedFileType,
                    promptForSaveLocation: true
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithSingleSelection);
            sandbox.stub(quickExport._UserSettings, "get").returns("rar");
            quickExport._psDialogs.promptForFile.returns(Q.resolve(expectedPath));

            return expect(quickExport._exportLayers(event, docinfoWithSingleSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(quickExport._psDialogs.promptForFile).to.be.calledWith("rar");
                });
        });

        it("should use the event setting for the dest folder if user setting doesn't exist", function () {
            // Pretend the user will select a different path.
            var expectedFile = _(docinfoWithSingleSelection.layers).first().name,
                expectedFileType = "png",
                expectedPath = destFolder + "/" + expectedFile + "." + expectedFileType,
                event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: expectedFileType,
                    promptForSaveLocation: true
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithSingleSelection);
            sandbox.stub(quickExport._UserSettings, "get").returns(undefined);
            quickExport._psDialogs.promptForFile.returns(Q.resolve(expectedPath));

            return expect(quickExport._exportLayers(event, docinfoWithSingleSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(quickExport._psDialogs.promptForFile).to.be.calledWith(destFolder);
                });
        });


        it("should export a multiple layer selection with a folder save prompt", function () {
            // Pretend the user will select a different path.
            var expectedFolder = "/Batcave/Pics/Gotham",
                event = {
                    name: PSEventStrings.QUICK_EXPORT_SELECTION,
                    destFolder: destFolder,
                    fileType: "png",
                    promptForSaveLocation: true
                },
                layers = DocinfoUtils.getLayersForExport(docinfoWithMultipleSelection);

            quickExport._psDialogs.promptForFolder.returns(Q.resolve(expectedFolder));

            return expect(quickExport._exportLayers(event, docinfoWithMultipleSelection, layers)).to.eventually.be.fulfilled
                .then(function () {
                    expect(quickExport._UserSettings.setCachedValue).to.be.calledWith("/Users/Batman/Desktop/gotham-skyline.psd", expectedFolder);
                    expect(BoundsUtils.ensureAccurateBoundsForLayers).to.have.been.calledWith(sinon.match(function (layers) {
                        return layers.length == docinfoWithMultipleSelection._selectionById.length;
                    }), docinfoWithMultipleSelection);
                    var expectedCalls = _(docinfoWithMultipleSelection.layers).pluck("name");
                    expect(FilePathSanitizer.sanitize).to.have.been.calledWith(expectedCalls[0]);
                    expect(FilePathSanitizer.sanitize).to.have.been.calledWith(expectedCalls[1]);
                    expect(FilePathSanitizer.sanitize).to.have.been.calledTwice;
                    expect(quickExport._psDialogs.promptForFolder).to.have.been.calledOnce;
                    expect(quickExport._buildComponent).to.have.been.calledTwice;
                    expect(quickExport._exportComponents).to.have.been.calledWith([
                        {
                            documentId: 3,
                            layerId: 7,
                            extension: "png",
                            path: path.join(expectedFolder, "batman.png")
                        },
                        {
                            documentId: 3,
                            layerId: 13,
                            extension: "png",
                            path: path.join(expectedFolder, "robin.png")
                        }
                    ]);
                });
        });
    });
});
